/*:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
::
::   Module      :   Angle Event Resource Framework API Header File
::   Copyright   :   (C)2002-2009 Woodward
::   Platform(s) :   MPC5xx
::   Limitations :   Dependant upon MPC5xx OS sub-project
::
:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::*/
/*! \file Resource_AngleEvent.h
 The BEHAVIOUR_ANGLE_EVENT behaviour provides angle event notifications. */

#ifndef __RESOURCE_ANGLEEVENT_H
#define __RESOURCE_ANGLEEVENT_H

/*----- INCLUDES ------------------------------------------------------------------------------------------*/
#include <typedefn.h>
#include <resource.h>

/*----- DEFINES -------------------------------------------------------------------------------------------*/

/*----- TYPEDEFS ------------------------------------------------------------------------------------------*/
#pragma pack(1)

/*! \brief Function pointer type that defines the prototype of the angle event notification function.

    The framework will call this function when the specified event is detected.
    
    The \p in_uNotifyData is application data that was supplied to the framework via the
    S_DiscreteOutCreateAttributes::uDiagCbackAppData when the callback function was defined. The function is
    defined as part of the CreateResourceBEHAVIOUR_DISCRETE_OUT() call. See \ref howtodiscretediagnostics for
    an example. */
typedef void (*AngleEventNotifyFuncPtrType)(NativePtrSizedInt_U in_uNotifyData);

/*! \brief Enumeration describes the set of mask definitions that are used to identify the valid discrete
           output attributes in the S_DiscreteOutAttributes and S_DiscreteOutCreateAttributes data structures.

Each mask definition applies to a particular member of one of the data structures.
\code
    ...
    S_DiscreteOutAttributes AttribsObj;

    // Going to provide values for the resource condition and report condition so include in the mask
    AttribsObj.uValidAttributesMask = USE_AE_CONDITION;
    AttribsObj.eResourceCondition = RES_ENABLED;
    // The remainder of the data structure can be left undefined since the uValidAttributesMask
    //   indicates that the members are not valid
    ... \endcode */
typedef enum
{
    USE_AE_NOTFIY_DATA = 0x01,              /*!< Selects S_AngleEventAttributes::uAppNotifyData */
    USE_AE_CONDITION = 0x02,                /*!< Selects S_AngleEventAttributes::eResourceCondition */
    USE_AE_NOTFIY_FUNC = 0x04,              /*!< Selects S_AngleEventAttributes::pfNotify */
    USE_AE_DYNAMIC_ON_CREATE = 0x08,        /*!< Selects S_AngleEventCreateAttributes::DynamicObj */
    USE_AE_ANGLE = 0x10,                    /*!< Selects S_AngleEventAttributes::s2NotifyAngle */
    USE_AE_DELAY = 0x20,                    /*!< Selects S_AngleEventAttributes::u2TriggerDelay */

/* IF THIS TYPE EXCEEDS 0x80 THEN ALTER THE TYPE OF THE uValidAttributesMask MEMBER(S) ACCORDINGLY */

} E_AngleEventAttributesMask;

/*! \brief This data structure describes all of the angle event's runtime configuration attributes.

    The attributes are altered through the use of SetResourceAttributesBEHAVIOUR_ANGLE_EVENT(). The data
    structure does not need to be completely filled inorder to be used. The \b uValidAttributesMask is a bit
    field member that is used to identify which attributes are valid. Each attribute is identified with a
    separate bit mask that is logic-ORed into the a mask when an attribute is to be used.
    \code
    ...
    S_AngleEventAttributes AEObj;

    // Going to affect the uAppNotifyData member and the condition 
    AEObj.uValidAttributesMask = USE_AE_NOTFIY_DATA | USE_AE_CONDITION;  \endcode */    
typedef struct
{
/*! Logic-OR the attributes [\ref USE_AE_CONDITION, \ref USE_AE_NOTFIY_DATA, \ref USE_AE_NOTFIY_FUNC,
    \ref USE_AE_DELAY, \ref USE_AE_ANGLE ]
    that are valid for this this instance of the data structure. */
    uint1 uValidAttributesMask;
/*! The condition of the resource. A disabled resource will not issue notfications even if the event has been specified
    of what the other runtime attributes have been set to. Diagnostic reports are also blocked when a resource
    is disabled. Use the \ref USE_AE_CONDITION mask to enable this attribute.  */
    E_ResourceCond eResourceCondition;
/*! The angle in degBfTDC#1 [16 = 1degBfTDC#1] that the application wants to be notified of. Use the \ref USE_AE_ANGLE
    mask to enable this attribute. This angle will be issued periodically unless the resource is disabled through the
    S_AngleEventAttributes::eResourceCondition member. */
    sint2 s2NotifyAngle;
/*! The time in microseconds from the occurrence of the notification angle until the notification is issued. Use the \ref USE_AE_DELAY
    mask to enable this attribute. This time delay is always applied. If no delay is required then set to zero. This attribute has the
    value of zero when the resource is first created. */
    uint2 u2TriggerDelay;
/*! The data that is to be supplied with the notification function that was specified within S_AngleEventAttributes::pfNotify.
    Use the \ref USE_AE_NOTFIY_DATA mask to enable this attribute. The previous value is maintained when not enabled. */
    NativePtrSizedInt_U uAppNotifyData;
/*! The notification function that is to be called by the framework when the requested event is observed. Use the
    \ref USE_AE_NOTFIY_FUNC mask to enable this attribute. This value defaults to NULL on creation. */
    AngleEventNotifyFuncPtrType pfNotify;

} S_AngleEventAttributes;

typedef S_AngleEventAttributes const* S_AngleEventAttributesPtr;

/*! \brief This data structure describes the creation attributes for an abngle event resource

    Included within the structure is the \b DynamicObj which represents the initial states for the resource's
    dynamic attributes. This data structure is referenced when the resource is created via the
    CreateResourceBEHAVIOUR_ANGLE_EVENT() function and \ref USE_AE_DYNAMIC_ON_CREATE is included in the
    S_AngleEventCreateAttributes::uValidAttributesMask member. */
typedef struct
{
/*! Logic-OR the attributes [\ref USE_AE_DYNAMIC_ON_CREATE that are valid for this this instance of the data structure. */
    uint1 uValidAttributesMask;

/*! Initial values of the runtime attributes, which can be later altered through a call to
    SetResourceAttributesBEHAVIOUR_ANGLE_EVENT(). Select with the \ref USE_AE_DYNAMIC_ON_CREATE bit mask */
    S_AngleEventAttributes DynamicObj;

} S_AngleEventCreateAttributes;

typedef S_AngleEventCreateAttributes const* S_AngleEventCreateAttributesPtr;

#pragma pack()
/*----- EXTERNALS -----------------------------------------------------------------------------------------*/

/*----- PROTOTYPES ----------------------------------------------------------------------------------------*/

NativeError_S CreateResourceBEHAVIOUR_ANGLE_EVENT(E_ModuleResource, S_AngleEventCreateAttributes const*);
NativeError_S SetResourceAttributesBEHAVIOUR_ANGLE_EVENT(E_ModuleResource, S_AngleEventAttributes const*);

#endif /* __RESOURCE_ANGLEEVENT_H */

/*----- END OF FILE ---------------------------------------------------------------------------------------*/
